<?php
require_once '../includes/auth.php';
requireAdminLogin();

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = $_POST['title'];
    $slug = $_POST['slug'];
    $description = $_POST['description'];
    
    // New Advanced Fields
    $enable_categories = isset($_POST['enable_categories']) ? 1 : 0;
    $category_label = $_POST['category_label'] ?? 'Select Category';
    $thank_you_message = $_POST['thank_you_message'] ?? '';
    $results_message = $_POST['results_message'] ?? '';
    
    // Basic Validation
    if (empty($title) || empty($slug)) {
        $error = "Title and Slug are required.";
    } elseif (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        $error = "Please upload a valid CSV file.";
    } else {
        // Check if slug exists
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM surveys WHERE slug = ?");
        $stmt->execute([$slug]);
        if ($stmt->fetchColumn() > 0) {
            $error = "Survey ID (Slug) already exists. Please choose a unique one.";
        } else {
            try {
                $pdo->beginTransaction();
                
                // 1. Create Survey
                $stmt = $pdo->prepare("INSERT INTO surveys (title, slug, description, enable_categories, category_label, thank_you_message, results_message) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$title, $slug, $description, $enable_categories, $category_label, $thank_you_message, $results_message]);
                $survey_id = $pdo->lastInsertId();
                
                // 2. Process CSV
                $handle = fopen($_FILES['csv_file']['tmp_name'], "r");
                $row = 0;
                
                // Cache for categories and parts to avoid redundant DB calls
                $categories_map = []; // Name -> ID
                $parts_map = []; // Name -> ID
                
                // Advanced Format Detection
                // If header row contains 'Category' or 'Part', use advanced mode
                $is_advanced_csv = false;
                $col_map = ['question' => 0]; // Default
                
                while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                    $row++;
                    
                    if ($row == 1) {
                        // Check Header
                        $header = array_map('strtolower', array_map('trim', $data));
                        
                        $cat_idx = array_search('category', $header);
                        $part_idx = array_search('part', $header);
                        $q_idx = array_search('question', $header);
                        
                        if ($cat_idx !== false || $part_idx !== false) {
                            $is_advanced_csv = true;
                            $col_map['category'] = $cat_idx;
                            $col_map['part'] = $part_idx;
                            $col_map['question'] = ($q_idx !== false) ? $q_idx : 0; // Fallback to 0 if not found but others are
                            
                            // Find option columns (anything that is not cat, part, or question)
                            $col_map['options'] = [];
                            foreach ($header as $idx => $col_name) {
                                if ($idx !== $cat_idx && $idx !== $part_idx && $idx !== $q_idx && !empty($col_name)) {
                                    $col_map['options'][] = $idx;
                                }
                            }
                            continue; // Skip header
                        } else {
                            // Standard format: Question, Opt1, Opt2...
                            // Assuming row 1 is header, skip it
                            continue;
                        }
                    }
                    
                    // Extract Data
                    if ($is_advanced_csv) {
                        $question_text = $data[$col_map['question']] ?? '';
                        $category_name = isset($col_map['category']) ? trim($data[$col_map['category']] ?? '') : '';
                        $part_name = isset($col_map['part']) ? trim($data[$col_map['part']] ?? '') : '';
                        
                        // Options
                        $options_data = [];
                        // If we identified option columns in header
                        if (!empty($col_map['options'])) {
                            foreach ($col_map['options'] as $idx) {
                                $options_data[] = $data[$idx] ?? '';
                            }
                        } else {
                            // Fallback: take all columns that are not Q, Cat, Part
                            foreach ($data as $idx => $val) {
                                if ($idx != $col_map['question'] && 
                                    ($idx != ($col_map['category'] ?? -1)) && 
                                    ($idx != ($col_map['part'] ?? -1))) {
                                    $options_data[] = $val;
                                }
                            }
                        }
                    } else {
                        // Standard Format
                        $question_text = $data[0] ?? '';
                        $category_name = '';
                        $part_name = '';
                        $options_data = array_slice($data, 1);
                    }
                    
                    if (empty($question_text)) continue;
                    
                    // Handle Category
                    $category_id = null;
                    if (!empty($category_name)) {
                        if (!isset($categories_map[$category_name])) {
                            $stmt = $pdo->prepare("INSERT INTO survey_categories (survey_id, name, sort_order) VALUES (?, ?, ?)");
                            $stmt->execute([$survey_id, $category_name, count($categories_map) + 1]);
                            $categories_map[$category_name] = $pdo->lastInsertId();
                        }
                        $category_id = $categories_map[$category_name];
                    }
                    
                    // Handle Part
                    $part_id = null;
                    if (!empty($part_name)) {
                        if (!isset($parts_map[$part_name])) {
                            $stmt = $pdo->prepare("INSERT INTO survey_parts (survey_id, title, sort_order) VALUES (?, ?, ?)");
                            $stmt->execute([$survey_id, $part_name, count($parts_map) + 1]);
                            $parts_map[$part_name] = $pdo->lastInsertId();
                        }
                        $part_id = $parts_map[$part_name];
                    }
                    
                    // Insert Question
                    $stmt = $pdo->prepare("INSERT INTO survey_questions (survey_id, question_text, sort_order, category_id, part_id) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([$survey_id, $question_text, $row, $category_id, $part_id]);
                    $question_id = $pdo->lastInsertId();
                    
                    // Insert Options
                    $option_sort = 1;
                    foreach ($options_data as $opt_text) {
                        $opt_text = trim($opt_text);
                        if (!empty($opt_text)) {
                            $stmt = $pdo->prepare("INSERT INTO survey_options (question_id, option_text, option_value, sort_order) VALUES (?, ?, ?, ?)");
                            $stmt->execute([$question_id, $opt_text, $opt_text, $option_sort++]);
                        }
                    }
                }
                
                fclose($handle);
                $pdo->commit();
                
                header("Location: surveys.php");
                exit;
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $error = "Error creating survey: " . $e->getMessage();
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Create Survey - BodhiTree Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600&display=swap" rel="stylesheet">
</head>
<body class="bg-gray-100 font-sans">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <div class="bg-slate-800 text-white w-64 flex-shrink-0">
            <div class="p-4 text-xl font-bold border-b border-slate-700">BodhiTree Admin</div>
            <nav class="mt-4">
                <a href="dashboard.php" class="block py-3 px-4 hover:bg-slate-700">Dashboard</a>
                <a href="surveys.php" class="block py-3 px-4 bg-slate-700 border-l-4 border-blue-500">Surveys</a>
                <a href="analytics.php" class="block py-3 px-4 hover:bg-slate-700">Analytics</a>
                <a href="mentors.php" class="block py-3 px-4 hover:bg-slate-700">Mentors</a>
                <a href="responses.php" class="block py-3 px-4 hover:bg-slate-700">Responses</a>
                <a href="freebies.php" class="block py-3 px-4 hover:bg-slate-700">Freebies</a>
                <a href="logout.php" class="block py-3 px-4 hover:bg-slate-700 text-red-300">Logout</a>
            </nav>
        </div>

        <!-- Content -->
        <div class="flex-1 overflow-auto p-8">
            <h1 class="text-3xl font-bold text-gray-800 mb-8">Create New Survey</h1>

            <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <div class="bg-white rounded-lg shadow p-8 max-w-2xl">
                <form method="POST" enctype="multipart/form-data" class="space-y-6">
                    <div>
                        <label class="block text-gray-700 text-sm font-bold mb-2">Survey Title</label>
                        <input type="text" name="title" required class="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="e.g. Student Feedback 2025">
                    </div>

                    <div>
                        <label class="block text-gray-700 text-sm font-bold mb-2">Unique ID (Slug)</label>
                        <input type="text" name="slug" required class="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="e.g. student_feedback_2025">
                        <p class="text-gray-500 text-xs mt-1">This will be used in the URL.</p>
                    </div>

                    <div>
                        <label class="block text-gray-700 text-sm font-bold mb-2">Description</label>
                        <textarea name="description" class="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" rows="3"></textarea>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">Custom Thank You Message</label>
                            <textarea name="thank_you_message" class="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" rows="3" placeholder="Default: Thank you for your support & Commitment..."></textarea>
                            <p class="text-gray-500 text-xs mt-1">Leave empty to use default.</p>
                        </div>
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">Custom Results Message</label>
                            <textarea name="results_message" class="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" rows="3" placeholder="Default: Do you want to know the Results..."></textarea>
                            <p class="text-gray-500 text-xs mt-1">Use {audience} placeholder for dynamic category name.</p>
                        </div>
                    </div>

                    <!-- Advanced Features Toggle -->
                    <div class="bg-blue-50 p-4 rounded-lg border border-blue-100">
                        <h3 class="font-bold text-blue-800 mb-2">Advanced Options (Optional)</h3>
                        
                        <div class="flex items-center mb-4">
                            <input type="checkbox" id="enable_categories" name="enable_categories" class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                            <label for="enable_categories" class="ml-2 block text-sm text-gray-900">
                                Enable User Categories (e.g. "Choose Grade")
                            </label>
                        </div>
                        
                        <div id="category_settings" class="hidden ml-6 mb-4">
                            <label class="block text-gray-700 text-sm font-bold mb-2">Category Selection Label</label>
                            <input type="text" name="category_label" value="Select Category" class="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="e.g. Select your Grade">
                        </div>
                    </div>

                    <script>
                        document.getElementById('enable_categories').addEventListener('change', function() {
                            document.getElementById('category_settings').classList.toggle('hidden', !this.checked);
                        });
                    </script>

                    <div>
                        <label class="block text-gray-700 text-sm font-bold mb-2">Upload CSV File</label>
                        <div class="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center hover:border-blue-500 transition-colors">
                            <input type="file" name="csv_file" required accept=".csv" class="hidden" id="csv_upload">
                            <label for="csv_upload" class="cursor-pointer text-blue-600 hover:text-blue-800">
                                <span class="block text-3xl mb-2">📄</span>
                                Click to upload CSV
                            </label>
                            <p id="file-name" class="text-gray-500 text-sm mt-2"></p>
                        </div>
                        <div class="mt-4 text-sm text-gray-600 bg-gray-50 p-4 rounded">
                            <p class="font-bold">Supported CSV Formats:</p>
                            <ul class="list-disc ml-5 mt-1 space-y-1">
                                <li><strong>Standard:</strong> <code>Question, Option 1, Option 2, ...</code></li>
                                <li><strong>Advanced:</strong> Include headers <code>Category</code> and/or <code>Part</code> to group questions.</li>
                                <li><em>Example Header:</em> <code>Category, Part, Question, Option A, Option B</code></li>
                            </ul>
                        </div>
                    </div>

                    <script>
                        document.getElementById('csv_upload').addEventListener('change', function() {
                            const fileName = this.files[0] ? this.files[0].name : '';
                            document.getElementById('file-name').textContent = fileName;
                        });
                    </script>

                    <div class="pt-4">
                        <button type="submit" class="w-full bg-blue-600 text-white font-bold py-3 rounded-lg hover:bg-blue-700 transition duration-200">
                            Create Survey
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</body>
</html>
